Below is the **core idea** of the solution architecture in Markdown format (`.md`). This document outlines the high-level design, components, and flow of the system.

---

# **Development Architecture for Universal File Upload System**

## **Overview**
The solution implements a **universal file upload system** that dynamically determines the storage path based on entity type (`ref`) and entity ID (`ref_id`). It ensures secure access using JWT authentication and restricts users to their own data. The system is designed to be flexible, scalable, and modular.

---

## **Core Idea**
The architecture revolves around the following principles:
1. **Dynamic File Handling**:
   - Files are stored in directories based on `ref` (e.g., `exhibition`, `booth`) and `ref_id` (e.g., `123`, `456`).
   - Example: `uploads/:ref/:ref_id/`.

2. **Ownership Validation**:
   - Each entity (e.g., exhibition, booth) is associated with a user (`userId`).
   - The system validates ownership by checking if the authenticated user owns the entity before allowing uploads.

3. **Universal Upload Endpoint**:
   - A single `/upload` endpoint handles all file uploads.
   - The endpoint uses `ref` and `ref_id` to determine the storage location and validate ownership.

4. **Metadata Storage**:
   - File paths and other metadata are stored in MongoDB for easy retrieval and management.

5. **Security**:
   - JWT tokens are used for authentication.
   - Input sanitization and ownership validation prevent unauthorized access.

---

## **Architecture Diagram**
```plaintext
+-------------------+       +-------------------+       +-------------------+
|                   |       |                   |       |                   |
|   Client (User)   +------>+   Express Server  +------>+   MongoDB Database |
|                   |       |                   |       |                   |
+-------------------+       +-------------------+       +-------------------+
                                |       ^                       ^
                                |       |                       |
                                v       |                       |
                         +-------------------+           +-------------------+
                         |                   |           |                   |
                         |   File Storage    |           |   Entity Models   |
                         |   (uploads/)      |           |   (Exhibition,    |
                         |                   |           |   Booth, etc.)    |
                         +-------------------+           +-------------------+
```

---

## **Components**

### **1. Client-Side**
- **Responsibilities**:
  - Send authenticated requests to the server.
  - Include `ref`, `ref_id`, and files in the request payload.
  - Handle responses (e.g., success or error messages).

- **Key Features**:
  - JWT token included in the `Authorization` header.
  - Support for multipart/form-data for file uploads.

---

### **2. Express Server**
- **Responsibilities**:
  - Authenticate users using JWT.
  - Validate ownership of entities (`ref_id`).
  - Dynamically determine file storage paths based on `ref` and `ref_id`.
  - Save file metadata in MongoDB.

- **Key Features**:
  - **Authentication Middleware**:
    - Validates JWT tokens and attaches the user object to the request.
  - **Universal Upload Endpoint**:
    - Accepts `ref`, `ref_id`, and files.
    - Uses `switch-case` to validate `ref` and map fields.
  - **File Handling**:
    - Uses `multer` for temporary file storage.
    - Moves files to the appropriate directory (`uploads/:ref/:ref_id/`).

---

### **3. MongoDB Database**
- **Responsibilities**:
  - Store metadata for entities (e.g., exhibitions, booths).
  - Associate entities with users (`userId`).
  - Store file paths for uploaded files.

- **Entity Models**:
  - **User**:
    ```javascript
    {
      _id: ObjectId,
      username: String,
      password: String
    }
    ```
  - **Exhibition**:
    ```javascript
    {
      _id: ObjectId,
      userId: ObjectId,
      name: String,
      start_date: Date,
      end_date: Date,
      logo: String,
      video_walls: String
    }
    ```
  - **Booth**:
    ```javascript
    {
      _id: ObjectId,
      userId: ObjectId,
      exhibitionId: ObjectId,
      name: String,
      posters: String,
      monitors: String
    }
    ```

---

### **4. File Storage**
- **Directory Structure**:
  - Files are stored in dynamically created directories:
    ```
    uploads/
    ├── exhibition/
    │   ├── 123/
    │   │   ├── logo.png
    │   │   └── video.mp4
    ├── booth/
    │   ├── 456/
    │   │   ├── poster.jpg
    │   │   └── monitor.png
    ```

- **Responsibilities**:
  - Store uploaded files in the correct directory based on `ref` and `ref_id`.
  - Ensure directories are created if they don’t exist.

---

## **Flow of Operations**

### **1. User Authentication**
1. The user logs in and receives a JWT token.
2. The token is included in the `Authorization` header for subsequent requests.

### **2. File Upload**
1. The client sends a POST request to `/upload` with:
   - `ref`: Entity type (e.g., `exhibition`, `booth`).
   - `ref_id`: Entity ID (e.g., `123`, `456`).
   - Files: Uploaded files (e.g., `logo`, `video_walls`).

2. The server:
   - Authenticates the user using the JWT token.
   - Validates ownership of the entity (`ref_id`).
   - Determines the target directory (`uploads/:ref/:ref_id/`).
   - Moves files to the target directory.
   - Updates the database with file paths.

3. The server responds with a success message and updated entity data.

  - types of refs that i will have 
  organizations
  users
  exhibitions
  boughten_booths
  video_walls
  posters   
  monitors  
  stands
  agents

---

## **Security Considerations**
1. **JWT Authentication**:
   - Ensures only authenticated users can upload files.
   - Tokens expire after a set duration (e.g., 1 hour).

2. **Ownership Validation**:
   - Prevents users from accessing or modifying entities they do not own.

3. **Input Sanitization**:
   - Sanitize `ref` and `ref_id` to prevent directory traversal attacks.
   - Validate file types and sizes to prevent malicious uploads.

4. **Error Handling**:
   - Gracefully handle errors (e.g., invalid `ref`, missing files).

---

## **Scalability**
1. **Cloud Storage**:
   - For large-scale applications, replace local storage with cloud services like AWS S3 or Google Cloud Storage.

2. **Database Indexing**:
   - Use indexes on `userId` and `ref_id` for faster queries.

3. **Load Balancing**:
   - Use a load balancer to distribute traffic across multiple server instances.

---

## **Conclusion**
This architecture provides a robust and flexible solution for handling file uploads dynamically based on `ref` and `ref_id`. By combining JWT authentication, ownership validation, and dynamic file storage, the system ensures security, scalability, and ease of use.